#include "LiquidCrystalRus.h"

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <avr/pgmspace.h>

#if defined(ARDUINO) && ARDUINO >= 100
  #include "Arduino.h"
#else
  #include "WProgram.h"
#endif

// это - российский перевод алфавита
// кроме 0401 --> 0xa2 = ╗, 0451 --> 0xb5
const PROGMEM uint8_t utf_recode[] = 
       { 0x41,0x80,0x42,0x92,0x81,0x45,0x82,0x83,0x84,0x85,0x4b,0x86,0x4d,0x48,0x4f,0x87,  // d0a
  //          A    Б    B    Г    Д    E    Ж    З    И    Й    K    Л    M    H    O    П
             0x50,0x43,0x54,0x88,0xd8,0x58,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x62,0x8f,0xac,0xad,  // d0b
  //          P    C    T    У    Ф    X    Ч    ч    Ш    Щ    Ъ    Ы    b    Э    Ю    Я
             0x61,0x36,0x42,0x92,0x81,0x65,0x82,0xb3,0x84,0x85,0x6b,0x86,0x4d,0x48,0x6f,0x87,  // d18
  //          a    6    B    Г    Д    e    Ж    ³    И    Й    k    Л    M    H    o    П
             0x70,0x63,0x54,0x79,0xd8,0x78,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x62,0x8f,0xac,0xad   // d19
  //          p    c    T    y    Ф    x    Ч    ч    Ш    Щ    Ъ    Ы    b    Э    Ю    Я
               
        };     

//Когда дисплей включается, он конфигурируется следующим образом:
//
// 1. Чистый дисплей
// 2. Функциональный набор: 
//    DL = 1; 8-разрядные интерфейсные данные
//    N = 0; 1-строчка дисплея
//    F = 0; 5x8 отметьте точкой символьный шрифт
// 3. Контроль откл/вкл дисплея: 
//    D = 0; Дисплей выключен 
//    C = 0; Курсор выключен 
//    B = 0; Мигание выключено
// 4. Режим установки: 
//    I/D = 1; инкремент 1 
//    S = 0; без сдвига 
//
//Примечание, однако, сброс Arduino не сбрасывает жидкокристаллический дисплей, таким образом, мы
//не может предположить, что находится в том состоянии, когда эскиз запускается (и вызывается LiquidCrystal).
//измененный 27 июля 2011
//Ильей В. Даниловым http://mk90 .ru/


LiquidCrystalRus::LiquidCrystalRus(uint8_t rs, uint8_t rw, uint8_t enable,
			     uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3,
			     uint8_t d4, uint8_t d5, uint8_t d6, uint8_t d7)
{
  init(0, rs, rw, enable, d0, d1, d2, d3, d4, d5, d6, d7);
}

LiquidCrystalRus::LiquidCrystalRus(uint8_t rs, uint8_t enable,
			     uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3,
			     uint8_t d4, uint8_t d5, uint8_t d6, uint8_t d7)
{
  init(0, rs, 255, enable, d0, d1, d2, d3, d4, d5, d6, d7);
}

LiquidCrystalRus::LiquidCrystalRus(uint8_t rs, uint8_t rw, uint8_t enable,
			     uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3)
{
  init(1, rs, rw, enable, d0, d1, d2, d3, 0, 0, 0, 0);
}

LiquidCrystalRus::LiquidCrystalRus(uint8_t rs,  uint8_t enable,
			     uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3)
{
  init(1, rs, 255, enable, d0, d1, d2, d3, 0, 0, 0, 0);
}

void LiquidCrystalRus::init(uint8_t fourbitmode, uint8_t rs, uint8_t rw, uint8_t enable,
			 uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3,
			 uint8_t d4, uint8_t d5, uint8_t d6, uint8_t d7)
{
  _rs_pin = rs;
  _rw_pin = rw;
  _enable_pin = enable;
  
  _data_pins[0] = d0;
  _data_pins[1] = d1;
  _data_pins[2] = d2;
  _data_pins[3] = d3; 
  _data_pins[4] = d4;
  _data_pins[5] = d5;
  _data_pins[6] = d6;
  _data_pins[7] = d7; 

  pinMode(_rs_pin, OUTPUT);
  // мы можем сохранить 1 контакт, не используя RW. Укажите, передающий 255 вместо pin#
  if (_rw_pin != 255) { 
    pinMode(_rw_pin, OUTPUT);
  }
  pinMode(_enable_pin, OUTPUT);
  
  if (fourbitmode)
    _displayfunction = LCD_4BITMODE | LCD_1LINE | LCD_5x8DOTS;
  else 
    _displayfunction = LCD_8BITMODE | LCD_1LINE | LCD_5x8DOTS;
  
  begin(16, 1);  
}

void LiquidCrystalRus::begin(uint8_t cols, uint8_t lines, uint8_t dotsize) {
  if (lines > 1) {
    _displayfunction |= LCD_2LINE;
  }
  _numlines = lines;
  _currline = 0;

  // приблизительно для 1 лнии дисплея Вы можете выбрать шрифт 10 пикселей высотой
  if ((dotsize != 0) && (lines == 1)) {
    _displayfunction |= LCD_5x10DOTS;
  }

  // ПОСМОТРИТЕ СТРАНИЦУ 45/46 ДЛЯ СПЕЦИФИКАЦИИ ИНИЦИАЛИЗАЦИИ!
  //согласно таблице данных, нам нужны по крайней мере 40 мс после того, как питание повысится выше 2.7 В
  //перед передающими командами. Ардуино может включить путь перед 4.5 В, таким образом, мы будем ожидать 50
  delayMicroseconds(50000); 
  digitalWrite(_rs_pin, LOW);
  digitalWrite(_enable_pin, LOW);
  if (_rw_pin != 255) { 
    digitalWrite(_rw_pin, LOW);
  }
  
  //переведите жидкокристаллический дисплей в режим на 8 битов или на 4 бита
  if (! (_displayfunction & LCD_8BITMODE)) {
    // это согласно таблице данных Hitachi HD44780
    // рисунок 24, pg 46

    // мы запускаем в режиме на 8 битов, пытаемся установить режим на 4 бита
    writeNbits(0x03,4);
    delayMicroseconds(4500); // ожидайте минута 4.1 мс

    // вторая попытка
    writeNbits(0x03,4);
    delayMicroseconds(4500); // ожидайте минута 4.1 мс
    
    // третья!
    writeNbits(0x03,4); 
    delayMicroseconds(150);

    // наконец, установлен 8-разрядный интерфейс
    writeNbits(0x02,4); 
  } else {
    // это согласно таблице данных Hitachi HD44780
    //рисунок 23 страницы 45
    
    // Отправьте функциональную последовательность набора команд
    command(LCD_FUNCTIONSET | _displayfunction);
    delayMicroseconds(4500);  // ожидайте больше чем 4.1 мс

    // вторая попытка
    command(LCD_FUNCTIONSET | _displayfunction);
    delayMicroseconds(150);

    // третья попытка
    command(LCD_FUNCTIONSET | _displayfunction);
  }

  // наконец, установите # серии, размер шрифта, и т.д.
  command(LCD_FUNCTIONSET | _displayfunction);  

  // включите дисплей без курсора или мерцающего значения по умолчанию
  _displaycontrol = LCD_DISPLAYON | LCD_CURSOROFF | LCD_BLINKOFF;  
  display();

  // очистить всё
  clear();

  // Инициализируйте по умолчанию текстовую папку (для романских языков)
  _displaymode = LCD_ENTRYLEFT | LCD_ENTRYSHIFTDECREMENT;
  // установите режим записи
  command(LCD_ENTRYMODESET | _displaymode);

}

void LiquidCrystalRus::setDRAMModel(uint8_t model) {
  _dram_model = model;
}

/********** команды высокого уровня, для пользователя! */
void LiquidCrystalRus::clear()
{
  command(LCD_CLEARDISPLAY);  // очистите дисплей, обнулите позицию курсора
  delayMicroseconds(2000);  // эта команда занимает много времени!
}

void LiquidCrystalRus::home()
{
  command(LCD_RETURNHOME);  // обнулите позицию курсора
  delayMicroseconds(2000);  // эта команда занимает много времени!
}

void LiquidCrystalRus::setCursor(uint8_t col, uint8_t row)
{
  int row_offsets[] = { 0x00, 0x40, 0x14, 0x54 };
  if ( row >= _numlines ) {
    row = _numlines-1;    // мы считаем строки, запускающиеся w/0
  }
  
  command(LCD_SETDDRAMADDR | (col + row_offsets[row]));
}

// Дисплей вкл\выкл (быстро)
void LiquidCrystalRus::noDisplay() {
  _displaycontrol &= ~LCD_DISPLAYON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}
void LiquidCrystalRus::display() {
  _displaycontrol |= LCD_DISPLAYON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// курсор подстрочный вкл\выкл
void LiquidCrystalRus::noCursor() {
  _displaycontrol &= ~LCD_CURSORON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}
void LiquidCrystalRus::cursor() {
  _displaycontrol |= LCD_CURSORON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// Включите и выключите мерцающий курсор
void LiquidCrystalRus::noBlink() {
  _displaycontrol &= ~LCD_BLINKON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}
void LiquidCrystalRus::blink() {
  _displaycontrol |= LCD_BLINKON;
  command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// Эти команды прокручивают дисплей, не изменяя оперативную память
void LiquidCrystalRus::scrollDisplayLeft(void) {
  command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVELEFT);
}
void LiquidCrystalRus::scrollDisplayRight(void) {
  command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVERIGHT);
}

// Это для текста, который идет Слева направо
void LiquidCrystalRus::leftToRight(void) {
  _displaymode |= LCD_ENTRYLEFT;
  command(LCD_ENTRYMODESET | _displaymode);
}

// Это для текста, который идет Справа налево
void LiquidCrystalRus::rightToLeft(void) {
  _displaymode &= ~LCD_ENTRYLEFT;
  command(LCD_ENTRYMODESET | _displaymode);
}

// Это 'выравнивает правую часть' текст по курсору
void LiquidCrystalRus::autoscroll(void) {
  _displaymode |= LCD_ENTRYSHIFTINCREMENT;
  command(LCD_ENTRYMODESET | _displaymode);
}

// Это 'выравнивает левую часть' текст по курсору
void LiquidCrystalRus::noAutoscroll(void) {
  _displaymode &= ~LCD_ENTRYSHIFTINCREMENT;
  command(LCD_ENTRYMODESET | _displaymode);
}

// Позволяет нам заполнять первые 8 расположений CGRAM
//пользовательскими знаками
void LiquidCrystalRus::createChar(uint8_t location, uint8_t charmap[]) {
  location &= 0x7; // у нас есть только 8 расположений 0-7
  command(LCD_SETCGRAMADDR | (location << 3));
  for (int i=0; i<8; i++) {
    write(charmap[i]);
  }
}

/*********** средний уровень команд, для отправки data/cmds */

inline void LiquidCrystalRus::command(uint8_t value) {
  send(value, LOW);
}

#if defined(ARDUINO) && ARDUINO >= 100
  size_t LiquidCrystalRus::write(uint8_t value)
#else
  void   LiquidCrystalRus::write(uint8_t value)
#endif
{
  uint8_t out_char=value;

  if (_dram_model == LCD_DRAM_WH1601) {  
    uint8_t ac=recv(LOW) & 0x7f;
    if (ac>7 && ac<0x14) command(LCD_SETDDRAMADDR | (0x40+ac-8));
  }

  if (value>=0x80) { // Обработка UTF-8
    if (value >= 0xc0) {
      utf_hi_char = value - 0xd0;
    } else {
      value &= 0x3f;
      if (!utf_hi_char && (value == 1)) 
        send(0xa2,HIGH); // ╗
      else if ((utf_hi_char == 1) && (value == 0x11)) 
        send(0xb5,HIGH); // ╦
      else 
        send(pgm_read_byte_near(utf_recode + value + (utf_hi_char<<6) - 0x10), HIGH);
    }    
  } else send(out_char, HIGH);
#if defined(ARDUINO) && ARDUINO >= 100
  return 1; // Принять успех
#endif
}

/************ низкоуровневые команды продвижения данных **********/

// запишите команду или данные автоматическим 4/8-разрядным выбором
void LiquidCrystalRus::send(uint8_t value, uint8_t mode) {
  digitalWrite(_rs_pin, mode);

  // если есть обозначенный контакт RW, установите его на low Write
  if (_rw_pin != 255) { 
    digitalWrite(_rw_pin, LOW);
  }
  
  if (_displayfunction & LCD_8BITMODE) {
    writeNbits(value,8); 
  } else {
    writeNbits(value>>4,4);
    writeNbits(value,4);
  }
}

// считайте данные автоматически 4/8-разрядным выбором
uint8_t LiquidCrystalRus::recv(uint8_t mode) {
  uint8_t retval;
  digitalWrite(_rs_pin, mode);

  // если есть контакт RW, установите его low to Write
  if (_rw_pin != 255) { 
    digitalWrite(_rw_pin, HIGH);
  }
  
  if (_displayfunction & LCD_8BITMODE) {
    retval = readNbits(8); 
  } else {
    retval = readNbits(4) << 4;
    retval |= readNbits(4);
  }
  return retval;
}
void LiquidCrystalRus::pulseEnable() {
  digitalWrite(_enable_pin, LOW);
  delayMicroseconds(1);    
  digitalWrite(_enable_pin, HIGH);
  delayMicroseconds(1);    // включающий импульс должен быть >450ns
  digitalWrite(_enable_pin, LOW);
  delayMicroseconds(100);   // для команд нужно > 37us 
}

void LiquidCrystalRus::writeNbits(uint8_t value, uint8_t n) {
  for (int i = 0; i < n; i++) {
    pinMode(_data_pins[i], OUTPUT);
    digitalWrite(_data_pins[i], (value >> i) & 0x01);
  }

  pulseEnable();
}

uint8_t LiquidCrystalRus::readNbits(uint8_t n) {
  uint8_t retval=0;
  for (int i = 0; i < n; i++) {
    pinMode(_data_pins[i], INPUT);
  }

  digitalWrite(_enable_pin, LOW);
  delayMicroseconds(1);    
  digitalWrite(_enable_pin, HIGH);
  delayMicroseconds(1);    // включающий импульс должен быть >450ns
  
  for (int i = 0; i < n; i++) {
    retval |= (digitalRead(_data_pins[i]) == HIGH)?(1 << i):0;
  }

  digitalWrite(_enable_pin, LOW);

  return retval;
}

